/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2012 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
******************************************************************************
*
* THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR 
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
* IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
* IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
* THE POSSIBILITY OF SUCH DAMAGE.
*
**************************************************************************/

#ifdef __CSMC__
#include "S12P64.h"
#else
#include <hidef.h>		/* common defines and macros */
#include "derivative.h"		/* derivative-specific definitions */
#endif
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "EEEGlobals.h"
#include "EEE.h"

volatile EEE_State EEEState = EEE_Idle;
volatile EEE_Error EEEError;
static DFSectorP EEEStartP;

/*************************************************************************************/

static void LaunchDFCmd(unsigned int *FCCOBData, unsigned char CmdLen, EEE_State State)

 {

 /* Variable Declarations */
 
 unsigned char x;

 /* Begin Function LaunchDFCmd() */
 
 FSTAT = FSTAT_ACCERR_MASK | FSTAT_FPVIOL_MASK;		/* clear errors from any previous commands */

 for (x = 0; x < CmdLen; x++)
  {
   FCCOBIX = x;
   FCCOB = *FCCOBData++;
  }

 EEEState = State;

 FSTAT = FSTAT_CCIF_MASK;	/* launch command */
  
 FCNFG_CCIE = 1;		/* enable Flash interrupt to clean up when command is finifhed */

 }	/* end LaunchDFCmd */

/*************************************************************************************/

static void DFBlankCheck(unsigned int DFAddress, unsigned int NumWords, Boolean Asynch)

 {

 /* Variable Declarations */

 unsigned int FCCOBBuff[3];
 
 /* Begin Function DFBlankCheck() */
 
 FCCOBBuff[0] = (unsigned int)((DFEraseVerfCmd << 8) + (((GlobalOffset + DFAddress) >> 16) & 0x00ff));
 FCCOBBuff[1] = (GlobalOffset + DFAddress) & 0xffff;;
 FCCOBBuff[2] = NumWords;
 
 LaunchDFCmd(FCCOBBuff, 3, EEE_BlankCheck);

 if (Asynch == FALSE)
  while (EEEState != EEE_Idle)			/* just wait here for the command to complete */
   ;

 }	/* end DFBlankCheck */

/*************************************************************************************/

static void DFEraseSector(unsigned int DFAddress, Boolean Asynch)

 {

 /* Variable Declarations */

 unsigned int FCCOBBuff[2];
 
 /* Begin Function DFEraseSector() */
 
 FCCOBBuff[0] = (unsigned int)((DFEraseSectorCmd << 8) + (((GlobalOffset + DFAddress) >> 16) & 0x00ff));
 FCCOBBuff[1] = (GlobalOffset + DFAddress) & 0xffff;;
 
 LaunchDFCmd(FCCOBBuff, 2, EEE_Erase);

 if (Asynch == FALSE)
  while (EEEState != EEE_Idle)			/* just wait here for the command to complete */
   ;

 }	/* end DFEraseSector */

/*************************************************************************************/

static void DFProgData(unsigned int DFAddress, unsigned int *Data, unsigned int NumWords, Boolean Asynch)

 {

 /* Variable Declarations */

 unsigned int FCCOBBuff[6];
 unsigned int x;
 
 /* Begin Function DFProgData() */
 
 FCCOBBuff[0] = (unsigned int)((DFProgramCmd << 8) + (((GlobalOffset + DFAddress) >> 16) & 0x00ff));
 FCCOBBuff[1] = (GlobalOffset + DFAddress) & 0xffff;
 
 for (x = 2; x < NumWords + 2; x++)
  {
   FCCOBBuff[x] = *Data++;
  }
 
 LaunchDFCmd(FCCOBBuff, NumWords + 2, EEE_ProgData);

 if (Asynch == FALSE)
  while (EEEState != EEE_Idle)			/* just wait here for the command to complete */
   ;

 }	/* end DFProgData */

/*************************************************************************************/

EEE_Error EEEWrite(unsigned char EEEAddress, unsigned char Data)

 {

 /* Variable Declarations */


 /* Begin Function EEEWrite() */
 
 if (EEEState != EEE_Idle)
  return(EEE_Busy);

 if (EEEAddress == 0)
  {
   return(EEE_BadAddress);
  }
  
 if (EEEStartP->Data[EEEAddress - 1] != Data)		/* if the data in the EEE RAM buffer is != to the new data... */
  {
   EEEStartP->Data[EEEAddress - 1] = Data;		/* put the new data in the EEE RAM buffer */
   if ((EEEStartP->Status & DelayedWriteFlag) == 0)	/* if the deferred write bit is set, just return */
    {
     EEEState = EEE_RAMBufCopyInit;			/* Set the state to initialize copy of EEE RAM buffer to DFlash */
     FCNFG_CCIE = 1;					/* enable Flash interrupt. it does all the copy work. */
    }
   }

 return(EEE_noErr);

 }	/* end EEEWrite */

/*************************************************************************************/

EEE_Error EEERead(unsigned char EEEAddress, unsigned char *Data)

 {

 /* Variable Declarations */


 /* Begin Function EEERead() */
 
 if (EEEAddress == 0)
  {
   return(EEE_BadAddress);
  }

 *Data = *(unsigned char *)(EEERAMStart + EEEAddress);

 return(EEE_noErr);

 }	/* end EEERead */

/*************************************************************************************/

EEE_Error EEEEnableDelayedWrite(void)

 {

 /* Variable Declarations */


 /* Begin Function EEEEnableDelayedWrite() */
 
 if (EEEState != EEE_Idle)				/* can't call this function if the EEE is busy */
  return(EEE_Busy);

 EEEStartP->Status |= DelayedWriteFlag;			/* Set the delayed write flag */

 return(EEE_noErr);

 }	/* end EEEEnableDelayedWrite */

/*************************************************************************************/

EEE_Error EEEFlushBuffer(void)

 {

 /* Variable Declarations */


 /* Begin Function EEEFlushBuffer() */
 
 if (EEEState != EEE_Idle)				/* can't call this function if the EEE is busy */
  return(EEE_Busy);

 if ((EEEStartP->Status & DelayedWriteFlag) == 0)	/* nothing to flush out of the buffer if DelayedWriteFlag is not set */
  return(EEE_FlushBufferErr);

 EEEState = EEE_RAMBufCopyInit;				/* Set the state to initialize copy of EEE RAM buffer to DFlash */
 EEEStartP->Status &= ~DelayedWriteFlag;		/* all future writes via EEEWrite() will cause an immediate write to DFlash */
 FCNFG_CCIE = 1;					/* enable Flash interrupt. it does all the copy work. */

 return(EEE_noErr);

 }	/* end EEEFlushBuffer */

/*************************************************************************************/

EEE_Error EEEInit(const unsigned char *DefaultData)

 {

 /* Variable Declarations */
 
 unsigned int x;
 DFSectorP SectorP;
 EEE_Error Error;
 unsigned int SectorHeader;
 DFSectorP PrevSector;
 DFSectorP NextSector;
 DFSectorP ActiveSector;
 unsigned char ActiveSectorNum;

 /* Begin Function EEEInit() */
 
 FCLKDIV = (BusClock / 1000) - 1;			/* initialize the Flash clock divider */
 SectorP = (DFSectorP)DFStart;				/* pointer to the start of the first physical DFlash sector */
 EEEStartP = (DFSectorP)EEERAMStart;			/* pointer to the EEE RAM buffer */
 
 DFBlankCheck(DFStart, DFSize / 2, FALSE);
 
 if (EEEError != EEE_noErr)				/* if the DFlash is not erased */
  {
   for (x = 0; x < DFNumSectors; x++)			/* we need to search through all the sectors */
    {
     if (SectorP[x].Status == ActiveSectorFlag)		/* currently 'active' sector? */
      {
       ActiveSector = &SectorP[x];			/* Mark this as the currently active sector */
       ActiveSectorNum = (unsigned char)x;		/* active sector number */
       if (x == 0)
        PrevSector = &SectorP[DFMaxSectorNum];		/* pointer to the previous sector */
       else
        PrevSector = &SectorP[x - 1];			/* pointer to the previous sector */
       DFBlankCheck((unsigned int)PrevSector, DFSectorSize / 2, FALSE);	/* is the previous sector blank? */
       if (EEEError != EEE_noErr)			/* if the previous sector is not blank, it needs to be erased */
        DFEraseSector((unsigned int)PrevSector, FALSE);	/* erase the previous sector */

       if ((x + 1) > DFMaxSectorNum)
        NextSector = &SectorP[0];			/* pointer to the next sector */
       else
        NextSector = &SectorP[x + 1];			/* pointer to the next sector */

       if (NextSector->Status == ActiveSectorFlag)	/* even though we found a sector flagged as active, we must check the next sector */
        {
         ActiveSector = NextSector;			/* Mark the next sector as the currently active sector */
         if ((x + 1) > DFMaxSectorNum)
          ActiveSectorNum = 0;
         else
          ActiveSectorNum = x + 1;			/* active sector number */
         DFEraseSector((unsigned int)&SectorP[x], FALSE);/* erase the 'previous' sector */
         break;						/* break out of the for loop */
        }
       else						/* even if the next sector isn't marked 'active' we still need to blank check */
        {
         DFBlankCheck((unsigned int)NextSector, DFSectorSize / 2, FALSE);	/* is the next sector blank? */
         if (EEEError != EEE_noErr)			/* if the next sector is not blank, it needs to be erased */
          DFEraseSector((unsigned int)NextSector, FALSE);	/* erase the next sector */
         break;						/* break out of the for loop */
        }
      }
    }
   for (x = 0; x < DFSectorSize - 1; x++)		/* copy all the data from the active DFlash sector to EEE RAM */
    EEEStartP->Data[x] = ActiveSector->Data[x];
 
   EEEStartP->Status = ActiveSectorNum;
  }
 else							/* entire DFlash array is blank */
  {
   SectorHeader = (ActiveSectorFlag << 8) + 0xff;	/* data for sector header */
   DFProgData(DFStart, &SectorHeader, 1, FALSE);	/* program the data */
   if (EEEError != EEE_noErr)				/* any errors? */
    {
     Error = EEEError;					/* yes. save the error code */
     EEEError = EEE_noErr;				/* clear the error code */
     return(Error);					/* return the error */
    }
   ActiveSector = (DFSectorP)DFStart;			/* active sector is the first sector */
   ActiveSectorNum = 0;					/* active sector number */

   for (x = 0; x < DFSectorSize - 1; x++)		/* copy all the data from the active DFlash sector to EEE RAM */
    EEEStartP->Data[x] = ActiveSector->Data[x];
 
   EEEStartP->Status = ActiveSectorNum;			/* currently active sector */

  }
 
 /* if the active sector is the first in the DFlash array & all the data is 0xff we will initialize it with the default data */

 if (EEEStartP->Status == 0)				/* is the current sector the first sector in the array? */
  {
   for (x = 0; x < DFSectorSize - 1; x++)		/* perform a blank check on the data */
    if (EEEStartP->Data[x] != 0xff)
     return(EEE_noErr);					/* if even 1 byte is not blank, return */

   if (DefaultData != NULL)				/* pointer to default data provided? */
    {
     (void )EEEEnableDelayedWrite();			/* delay writes so we can write default data into the EEE */
   
     for (x = 1; x < DFSectorSize; x++)			/* write all 255 bytes */
      {
       Error = EEEWrite((unsigned char)x, DefaultData[x]);	/* write a byte into the EEE buffer */
       if (Error != EEE_noErr)					/* check for errors */
        {
         (void) EEEFlushBuffer();
         return(Error);
        }
      }
     (void) EEEFlushBuffer();				/* Write the buffer into the next available sector */
     while (EEEState != EEE_Idle)			/* just wait here for the writes to complete */
      ;
    }

  }

 return(EEE_noErr);					/* return */
 
 }	/* end EEEInit */

/*************************************************************************************/
